'use client';

import { motion } from 'framer-motion';
import { useLanguage } from '@/contexts/language-context';
import {
  Globe,
  ShieldCheck,
  Cpu,
  CircuitBoard,
  Scaling,
  Shield,
  Zap,
  Wind,
  Signal,
  Wrench,
  ArrowRight,
} from 'lucide-react';
import React from 'react';
import { Button } from './ui/button';
import Link from 'next/link';

const iconMap: { [key: string]: React.ElementType } = {
  CircuitBoard,
  Scaling,
  Cpu,
  Zap,
  Shield,
  ShieldCheck,
  Globe,
  Wind,
  Signal,
};

const cardColors = [
  { gradient: 'from-blue-600 to-cyan-500', bg: 'bg-blue-600/10', glow: 'shadow-blue-500/50' },
  { gradient: 'from-emerald-600 to-green-500', bg: 'bg-emerald-600/10', glow: 'shadow-emerald-500/50' },
  { gradient: 'from-blue-700 to-teal-500', bg: 'bg-blue-700/10', glow: 'shadow-blue-700/50' },
  { gradient: 'from-green-600 to-lime-500', bg: 'bg-green-600/10', glow: 'shadow-green-600/50' },
  { gradient: 'from-cyan-600 to-sky-500', bg: 'bg-cyan-600/10', glow: 'shadow-cyan-500/50' },
  { gradient: 'from-teal-600 to-emerald-500', bg: 'bg-teal-600/10', glow: 'shadow-teal-500/50' },
  { gradient: 'from-blue-800 to-indigo-600', bg: 'bg-blue-800/10', glow: 'shadow-blue-800/50' },
  { gradient: 'from-green-700 to-teal-600', bg: 'bg-green-700/10', glow: 'shadow-green-700/50' },
  { gradient: 'from-cyan-700 to-blue-600', bg: 'bg-cyan-700/10', glow: 'shadow-cyan-700/50' },
];

export default function NoorServicesSection() {
  const { content } = useLanguage();
  const { noorServices } = content;

  return (
    <section className="relative overflow-hidden bg-background py-20 sm:py-24">
      {/* Background Grid */}
      <div className="absolute inset-0 z-0">
        <div className="absolute inset-0 bg-[radial-gradient(#e2e8f0_1px,transparent_1px)] [background-size:32px_32px] dark:bg-[radial-gradient(#334155_1px,transparent_1px)]"></div>
        <div className="absolute inset-0 bg-gradient-to-b from-background via-background/90 to-background"></div>
      </div>
      
      <div className="container relative z-10 mx-auto px-4">
        {/* Header */}
        <motion.div
          initial={{ opacity: 0, y: -20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.5 }}
          className="mx-auto mb-16 max-w-3xl text-center"
        >
          <div className="mb-4 inline-flex items-center gap-2 rounded-full border border-primary/10 bg-primary/5 px-4 py-2">
            <Wrench className="h-5 w-5 text-green-500" />
            <p className="text-sm font-semibold text-primary">{noorServices.badge}</p>
          </div>
          <h2 className="text-4xl font-bold tracking-tight bg-gradient-to-r from-primary via-green-500 to-primary bg-clip-text text-transparent sm:text-5xl">
            {noorServices.title}
          </h2>
          <p className="mt-6 text-lg leading-8 text-muted-foreground">
            {noorServices.subtitle}
          </p>
        </motion.div>

        {/* Features Grid */}
        <div className="grid grid-cols-1 gap-8 sm:grid-cols-2 lg:grid-cols-3">
          {noorServices.items.map((item, index) => {
            const Icon = iconMap[item.icon] || Wrench;
            const color = cardColors[index % cardColors.length];

            return (
              <motion.div
                key={index}
                initial={{ opacity: 0, scale: 0.9 }}
                whileInView={{ opacity: 1, scale: 1 }}
                viewport={{ once: true }}
                transition={{ duration: 0.5, delay: index * 0.1 }}
                className="group relative"
              >
                <div className={`absolute -inset-px rounded-2xl bg-gradient-to-r ${color.gradient} opacity-0 blur transition-opacity duration-500 group-hover:opacity-70`} />
                <div className={`relative h-full overflow-hidden rounded-2xl border border-border ${color.bg} p-8 transition-all duration-300 group-hover:bg-transparent`}>
                  <div className={`absolute -right-12 -top-12 h-32 w-32 rounded-full bg-gradient-to-r ${color.gradient} opacity-10 blur-2xl transition-opacity duration-500 group-hover:opacity-20`} />
                  
                  <div className="relative flex flex-col h-full">
                    <div className={`mb-6 inline-block rounded-xl bg-gradient-to-r ${color.gradient} p-4 shadow-lg ${color.glow} w-fit`}>
                      {Icon && <Icon className="h-8 w-8 text-white" />}
                    </div>
                    <h3 className="mb-2 text-xl font-bold text-primary">
                      {item.title}
                    </h3>
                    <p className="flex-grow text-muted-foreground">
                      {item.description}
                    </p>
                  </div>
                </div>
              </motion.div>
            );
          })}
        </div>

        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.5, delay: 0.5 }}
          className="mt-16 text-center"
        >
          <Button asChild size="lg" className="group bg-gradient-to-r from-blue-600 to-green-500 text-white hover:from-blue-700 hover:to-green-600">
            <Link href="https://www.noorenergy-iq.com/service2.html" target="_blank" rel="noopener noreferrer">
              {noorServices.learnMore}
              <ArrowRight className="h-5 w-5 transition-transform duration-300 group-hover:translate-x-1" />
            </Link>
          </Button>
        </motion.div>
      </div>
    </section>
  );
}

    